/******************************************************************************
 * (C) Copyright 1999 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xobserve.c
 *       Observer functions
 * -----------------------------------------------------------------*/

#include <xtypedef.h>
#include <xdynamic.h>
#include <xerrcapi.h>
#include <xiocommo.h>
#include <xobserve.h>
#include <xregcons.h>
#include <xsession.h>
#include <xaddrmap.h>
#include <xboard.h>
#include <xutil.h>
#include <xsetting.h>

 
/* Pointer to analyzer database */
#define DB_OBSERVER (bx_handlearray[handle].db->Analyzer.Observer)

/* Value of properties */
#define Mask(index)     (DB_OBSERVER.Mask[index])
#define AccuErr(index)  (DB_OBSERVER.AccuErr[index])
#define FirstErr(index) (DB_OBSERVER.FirstErr[index])

/*---------------------------------------------------------------------------*
 * BestXObsRuleGet
 *
 * Purpose: This function returns the rule for the given bitposition
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXObsRuleGet(
  bx_handletype handle,
  bx_int32 bitpos,
  bx_obsruletype *rule
)
{
  BX_DECLARE_FUNCNAME("BestXObsRuleGet");

  const bx_generic_infotype *geninfo;  /* hold the generic infotype for observer rules */
  const bx_param_infotype *parinfo;

  bx_errtype err,errt = BX_E_INVALID_OBS_RULE;  /* this is the default */
  bx_int32 i;

  BX_TRY_VARS_NO_PROG;
  
  BX_TRY_FCT_PARAM_NULL_POINTER(rule);
  
  BX_TRY_BEGIN 
  {
    BX_LICENSECHECK(BX_CAPABILITY_OBSERVER|BX_CAPABILITY_CAPI); 
    
    /* get the generic information about observer rules */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_OBSRULES, &geninfo));

    /* run through all rules and look for bitpos */
    /* geninfo->num_elem equals BX_OBSRULEPROP_SIZE */
    for (i = 0; i < geninfo->num_elem; i++)
    {
      err = BestXParamInfoGet(handle, BX_PARAM_OBSRULES, i, &parinfo,(bx_int32)BX_INDEX_SEARCH);
      if (err)
      {
        BestXLastErrorAllParamsSet(handle,(bx_int32)BX_PARAM_OBSRULES,i,0UL,0UL,0UL);
        BX_ERRCHECK(err);
      }
      
      if (parinfo->group==bitpos)
      {
        *rule=parinfo->proptyp.obsrule;
        errt=BX_E_OK;
        break;
      }
    }
  }
  BX_ERRETURN (errt);
}

/*---------------------------------------------------------------------------*
 * BestXObsBitPosGet
 *
 * Purpose: This function returns the bitposition for the given rule
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXObsBitPosGet(
  bx_handletype handle,
  bx_obsruletype rule,
  bx_int32 *bitpos
)
{
  BX_DECLARE_FUNCNAME("BestXObsBitPosGet");
  
  const bx_generic_infotype *geninfo;  /* hold the generic infotype for observer rules */
  const bx_param_infotype *parinfo;
  bx_int32 i;
  bx_errtype err;

  BX_TRY_VARS_NO_PROG;
  
  bx_errtype errt = BX_E_INVALID_OBS_RULE;  /* this is the default */

  BX_TRY_FCT_PARAM_NULL_POINTER(bitpos);
    
  BX_TRY_BEGIN 
  {
    /* get the generic information about observer rules */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_OBSRULES, &geninfo));

    /* run through all rules and look for rules */
    /* geninfo->num_elem equals BX_OBSRULEPROP_SIZE */
    for (i = 0; i < geninfo->num_elem; i++)
    {
      err = BestXParamInfoGet(handle, BX_PARAM_OBSRULES, i, &parinfo,(bx_int32)BX_INDEX_SEARCH);
      if (err)
      {
        BestXLastErrorAllParamsSet(handle,(bx_int32)BX_PARAM_OBSRULES,i,0UL,0UL,0UL);
        BX_ERRCHECK(err);
      }
      
      if (rule==parinfo->proptyp.obsrule)
      {
        *bitpos=parinfo->group; /* don't ever change this again! Otherwise change SVP also (PCIXCARD.CPP) HL */
        errt=BX_E_OK;
        break;
      }
    }
  }
  BX_ERRETURN (errt);
}

/*---------------------------------------------------------------------------*
 * BestXObsRuleInfoGet
 *
 * Purpose: This function returns rule infos as a string
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXObsRuleInfoGet(
  bx_handletype handle,
  bx_obsruletype rule,
  bx_int32 type,               /* 1=error description, 0=short rule name */ 
  bx_charptrtype * errortext
)
{
  BX_DECLARE_FUNCNAME("BestXObsRuleInfoGet [obsruleinfoget]");

  const bx_generic_infotype *geninfo;  /* hold the generic infotype for observer rules */
  const bx_param_infotype *parinfo;
  bx_int32 i;
  bx_errtype err;

  BX_TRY_VARS_NO_PROG;
  
  bx_errtype errt = BX_E_INVALID_OBS_RULE;  /* this is the default */

  BX_FCT_PARAM_NULL_POINTER_CHK(errortext);
    
  BX_TRY_BEGIN 
  {
    /* get the generic information about observer rules */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_OBSRULES, &geninfo));

    /* run through all rules and look for rule */
    for (i = 0; i < geninfo->num_elem; i++)
    {
      err = BestXParamInfoGet(handle, BX_PARAM_OBSRULES, i, &parinfo,(bx_int32)BX_INDEX_SEARCH);
      if (err)
      {
        BestXLastErrorAllParamsSet(handle,(bx_int32)BX_PARAM_OBSRULES,i,0UL,0UL,0UL);
        BX_ERRCHECK(err);
      }
      
      if (rule==parinfo->proptyp.obsrule)
      {
        *errortext=(type?parinfo->param_short/* complete description */ : parinfo->param_name /* Rule name */);
        errt=BX_E_OK;
        break;
      }
    }
  }
  BX_ERRETURN (errt);
}

/*---------------------------------------------------------------------------*
 * BestXObsRuleStringGet
 *
 * Purpose: This function returns a textual explanation for the given rule
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXObsRuleStringGet(
  bx_handletype handle,
  bx_obsruletype rule,
  bx_charptrtype * errortext
)
{
  BX_DECLARE_FUNCNAME("BestXObsErrStringGet [obserrstringget]");
  BX_TRY_VARS_NO_PROG;
    
  BX_TRY_BEGIN 
  {
    BX_LICENSECHECK(BX_CAPABILITY_OBSERVER|BX_CAPABILITY_CAPI); 
    BX_TRY(BestXObsRuleInfoGet(handle,rule,1 /* complete description */,errortext));
  }
  BX_ERRETURN (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXObsDefaultSet
 *
 * Purpose: Defaults the observer props in CAPI DB 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXObsDefaultSet(
  bx_handletype handle
)
{
  bx_int32 i;
  BX_DECLARE_FUNCNAME ("BestXObsDefaultSet [obsdefaultset]");
  
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {
    BX_LICENSECHECK(BX_CAPABILITY_OBSERVER|BX_CAPABILITY_CAPI); 

    for (i=0;i<BX_OBSRULEPROP_DW30;i++)
    {
      Mask(i)=0;      /* 0=not masked, i.e. rules enabled */      
      FirstErr(i)=0;  /* 0=no error occ. */
      AccuErr(i)=0;   /* 0=no error occ. */
    }
  }

  BX_ERRETURN (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXObsProg
 *
 * Purpose: Updates all mask values on card by copying them from host to card.
 *          To set mask values for specific rules, call BestXObsMaskSet() before.
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXObsProg(
  bx_handletype   handle
)
{
  bx_int32 NumRegs,MaskAddress,i;
  
  BX_DECLARE_FUNCNAME ("BestXObsProg [obsprog]");
  
  BX_TRY_VARS;

  BX_TRY_BEGIN 
  {
    BX_LICENSECHECK(BX_CAPABILITY_OBSERVER|BX_CAPABILITY_CAPI); 

    /* Determine number of registers to program */
    NumRegs=(BestXHasMephisto(handle)?BX_OBSRULEPROP_DW:BX_OBSRULEPROP_DW30);
    
    /* Determine start addresses */
    MaskAddress=(BestXHasMephisto(handle)?BX_REG_PROT_ERR_MASK_M:BX_REG_PROT_ERR_MASK_F);

    /* lock the resource */
    BX_TRY_PROGRESS(BestXResourceLock(handle, BX_RESLOCK_OBSERVER));

    for (i=0;i<NumRegs;i++)
    {
      /* Write mask to card; invert for HW */
      BX_TRY(BestXDirectRegWrite(handle,MaskAddress,sizeof(bx_int32),~Mask(i)));
      MaskAddress+=sizeof(bx_int32);
      
      /* Clear all errors on host DB (card errors are cleared below) */
      FirstErr(i)=0;
      AccuErr(i)=0;
    }

    /* Clear all errors in order to avoid wrong contents of the 
       first error register. */
    BX_TRY(BestXStatusClear(handle,BX_STAT_OBS_ERR));

    /* unlock the resource */
    BX_TRY_PROGRESS(BestXResourceUnlock(handle, BX_RESLOCK_OBSERVER));
  }

  BX_ERRETURN (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXObsRead
 *
 * Purpose: Updates all mask + error values on host by copying 
 *          them from card to host. After executing BestXObsRead() you can
 *          use the various Get() functions to retrieve further info
 *          about specific rules.
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXObsRead( 
  bx_handletype   handle
)
{
  bx_int32 MaskAddress,AccuErrAddress,FirstErrAddress,NumRegs,i;

  BX_DECLARE_FUNCNAME ("BestXObsRead [obsread]");
  
  BX_TRY_VARS;

  BX_TRY_BEGIN 
  {
    BX_LICENSECHECK(BX_CAPABILITY_OBSERVER|BX_CAPABILITY_CAPI); 

    /* Determine number of registers to read from */
    NumRegs=(BestXHasMephisto(handle)?BX_OBSRULEPROP_DW:BX_OBSRULEPROP_DW30);
    
    /* Determine start addresses */
    MaskAddress=     (BestXHasMephisto(handle)?BX_REG_PROT_ERR_MASK_M:BX_REG_PROT_ERR_MASK_F);
    AccuErrAddress=  (BestXHasMephisto(handle)?BX_REG_ACCUM_ERR_REG_M:BX_REG_ACCUM_ERR_REG_F);
    FirstErrAddress= (BestXHasMephisto(handle)?BX_REG_FIRST_ERR_M:BX_REG_FIRST_ERR_F);

    /* lock the resource */
    BX_TRY_PROGRESS(BestXResourceLock(handle, BX_RESLOCK_OBSERVER));

    /* We first have to write to the shadow strobe register here ! */
    if (BestXHasMephisto(handle))
    {
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_REG_SH_STRB_M,sizeof(bx_int16),0));
    }
    else
    {
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_OBS_CTRL_STATUS_REG_F,sizeof(bx_int16),1<<2));
    }

    for (i=0;i<NumRegs;i++)
    {
      /* Read mask */
      BX_TRY(BestXDirectRegRead(handle,MaskAddress,sizeof(bx_int32),&Mask(i)));
      Mask(i)=~Mask(i);  /* Invert from HW to SW */
      MaskAddress+=sizeof(bx_int32);

      /* Read accu error */ 
      BX_TRY(BestXDirectRegRead(handle,AccuErrAddress,sizeof(bx_int32),&AccuErr(i)));
      AccuErrAddress+=sizeof(bx_int32);

      /* Read first error */
      BX_TRY(BestXDirectRegRead(handle,FirstErrAddress,sizeof(bx_int32),&FirstErr(i)));
      FirstErrAddress+=sizeof(bx_int32);
    }

    /* unlock the resource */
    BX_TRY_PROGRESS(BestXResourceUnlock(handle, BX_RESLOCK_OBSERVER));
  }

  BX_ERRETURN (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXObsMaskSet
 *
 * Purpose: Sets mask value for a specific rule.
 *          This functions does not access the card.
 *          To update the mask info on card, call BestXObsWrite() afterwards.
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXObsMaskSet( 
  bx_handletype   handle,
  bx_obsruletype rule,
  bx_int32 mask                 /* 1=rule masked/disabled (0 in HW) */
)
{
  bx_int32 bitpos=0;

  BX_DECLARE_FUNCNAME ("BestXObsMaskSet [obsmaskset]");
  
  BX_TRY_VARS;

  BX_TRY_BEGIN 
  {
    BX_LICENSECHECK(BX_CAPABILITY_OBSERVER|BX_CAPABILITY_CAPI); 

    /* lock resource. */
    BX_TRY_PROGRESS(BestXResourceLock(handle,BX_RESLOCK_OBSERVER));

    /* Check if rule is allowed and value is in range.
       This does not check HW dependencies !! */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_OBSRULES, (bx_int32) rule, mask));
    
    /* Convert rule to bitposition */
    BX_TRY(BestXObsBitPosGet(handle,rule,&bitpos));    

    if (mask)
    {
      /* Disable rule */
      Mask(bitpos/32) |= 1<<(bitpos%32);
    }
    else
    {
      /* Enable rule */
      Mask(bitpos/32) &= ~(((bx_int32)1)<<(bitpos%32));
    }
  }

  BX_ERRETURN (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXObsMaskGet
 *
 * Purpose: Returns mask of a specific rule.
 *          This functions does not access the card.
 *          To update the mask info on host, call BestXObsRead() before.
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXObsMaskGet( 
  bx_handletype   handle,
  bx_obsruletype rule,
  bx_int32 *mask            
)
{
  bx_int32 bitpos=0;
  
  BX_DECLARE_FUNCNAME ("BestXObsMaskGet [obsmaskget]");
  
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {
    BX_TRY_FCT_PARAM_NULL_POINTER(mask);

    BX_LICENSECHECK(BX_CAPABILITY_OBSERVER|BX_CAPABILITY_CAPI); 
 
    /* Check if rule is allowed and value is in range.
       This does not check HW dependencies !! */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_OBSRULES, (bx_int32) rule,0));
    
    /* Convert rule to bitposition */
    BX_TRY(BestXObsBitPosGet(handle,rule,&bitpos));    

    *mask=((Mask(bitpos/32))&1<<bitpos%32?1:0);
  }

  BX_ERRETURN (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXObsErrorGet
 *
 * Purpose: Helper function; returns errors for given rule
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXObsErrorGet(
  bx_handletype  handle,
  bx_obsruletype rule,                   
  bx_int32 *accuerr,
  bx_int32 *firsterr                 
)
{
  bx_int32 bitpos=0;

  BX_DECLARE_FUNCNAME ("BestXObsErrorGet [obserrorget]");
  
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {
    BX_LICENSECHECK(BX_CAPABILITY_OBSERVER|BX_CAPABILITY_CAPI); 
 
    /* Check if rule is allowed and value is in range.
       This does not check HW dependencies !! */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_OBSRULES, (bx_int32) rule,0));
    
    /* Convert rule to bitposition */
    BX_TRY(BestXObsBitPosGet(handle,rule,&bitpos));    

    if (accuerr)
    {
      *accuerr= (AccuErr(bitpos/32) & ((bx_int32)1<<(bitpos%32)) ? 1 : 0);
    }
    if (firsterr)
    {
      *firsterr= (FirstErr(bitpos/32) & ((bx_int32)1<<(bitpos%32)) ? 1 : 0);
    }
  }

  BX_ERRETURN (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXObsAccuErrorGet
 *
 * Purpose: Returns accumulated error value for a specific rule.
 *          This functions does not access the card.
 *          To update the error info, call BestXObsRead() before resp. between 
 *          multiple calls to this function.
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXObsAccuErrorGet(  
  bx_handletype  handle,
  bx_obsruletype rule,                   
  bx_int32 *accuerr
)
{
  BX_DECLARE_FUNCNAME ("BestXObsAccuErrorGet [obsaccuerrorget]");
  
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {
    BX_LICENSECHECK(BX_CAPABILITY_OBSERVER|BX_CAPABILITY_CAPI); 
    BX_TRY_FCT_PARAM_NULL_POINTER(accuerr);
    
    BX_TRY(BestXObsErrorGet(handle,rule,accuerr,NULL));
  }

  BX_ERRETURN (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXObsFirstErrorGet
 *
 * Purpose: Returns first error value for a specific rule.
 *          This functions does not access the card.
 *          To update the error info, call BestXObsRead() before resp. 
 *          between multiple calls to this function.
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXObsFirstErrorGet(  
  bx_handletype  handle,
  bx_obsruletype rule,                   
  bx_int32 *firsterr
)
{
  BX_DECLARE_FUNCNAME ("BestXObsFirstErrorGet [obsfirsterrorget]");
  
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {
    BX_LICENSECHECK(BX_CAPABILITY_OBSERVER|BX_CAPABILITY_CAPI); 
    BX_TRY_FCT_PARAM_NULL_POINTER(firsterr);
    
    BX_TRY(BestXObsErrorGet(handle,rule,NULL,firsterr));
  }

  BX_ERRETURN (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXObsStatusGet
 *
 * Purpose: Returns textual error summary.
 *          This functions does not access the card.
 *          To update the error info, call BestXObsRead() before.
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXObsStatusGet( /* @obsstatusget */
  bx_handletype   handle,
  bx_charptrtype  *statustext    /* #RETURN "%s\n" */
)
{
  BX_DECLARE_FUNCNAME("BestXObsStatusGet [obsstatusget]");

  BX_TRY_VARS_NO_PROG;
  bx_obsruletype rule=(bx_obsruletype)0;
  bx_charptrtype rulename;
  static char buf[0x10000];
  bx_int32 bitpos=0,accuerr=0,firsterr=0,mask=0;

  BX_FCT_PARAM_NULL_POINTER_CHK(statustext);
    
  buf[0]='\0';
    
  BX_TRY_BEGIN 
  {
    BX_LICENSECHECK(BX_CAPABILITY_OBSERVER|BX_CAPABILITY_CAPI); 

    BESTX_SPRINTF(buf+BESTX_STRLEN(buf),"\nObserver Status\n");
    BESTX_SPRINTF(buf+BESTX_STRLEN(buf),"ERRORS:\n");
    
    for (bitpos=0;bitpos<BX_OBSRULEPROP_SIZE30;bitpos++)
    {
      BX_TRY(BestXObsRuleGet(handle,bitpos,&rule));
      BX_TRY(BestXObsAccuErrorGet(handle,rule,&accuerr));
      if (accuerr)
      {
        /* Rule rule has triggered */

        /* Get rule description: */
        BX_TRY(BestXObsRuleStringGet(handle,rule,&rulename));
        BESTX_SPRINTF(buf+BESTX_STRLEN(buf),"%s",rulename);

        /* Check mask */
        BX_TRY(BestXObsMaskGet(handle,rule,&mask));
        if (mask) BESTX_SPRINTF(buf+BESTX_STRLEN(buf)," (masked)");

        /* Check first error */
        BX_TRY(BestXObsFirstErrorGet(handle,rule,&firsterr));
        if (firsterr) BESTX_SPRINTF(buf+BESTX_STRLEN(buf)," (first)");

        BESTX_SPRINTF(buf+BESTX_STRLEN(buf),"\n");
      }
    }
  }

  *statustext=buf;  

  BX_ERRETURN (BX_TRY_RET);
}
